//package globalExamples;

import oracle.olapi.data.source.DataProvider;
import oracle.olapi.data.source.Source;
import oracle.olapi.data.source.StringSource;

import oracle.olapi.metadata.mdm.MdmAttribute;
import oracle.olapi.metadata.mdm.MdmLevelHierarchy;
import oracle.olapi.metadata.mdm.MdmMeasure;
import oracle.olapi.metadata.mdm.MdmPrimaryDimension;

/**
 * Complete code for the examples in the descriptions of the following methods 
 * of the <CODE>Source</CODE> class in the Oracle OLAP Java API Reference.
 * 
 * <P>
 * <UL>
 *   <LI>Example 1: 
 *       recursiveJoin(Source joined, Source comparison, Source parent,
 *                     int comparisonRule, boolean parentsFirst,
 *                     boolean parentsRestrictedToBase, int maxIterations,
 *                     boolean visible)
 *   <LI>Example 2: 
 *       recursiveJoin(Source joined, String comparison, Source parent, 
 *                     int comparisonRule)
 *   <LI>Example 3: 
 *       recursiveJoin(Source joined, String[] comparison, Source parent, 
 *                     int comparisonRule)
 *   <LI>Example 4: selectDescendants(Source comparison, Source parent)
 *   <LI>Example 5:
 *       sortDescendingHierarchically(Source joined, Source parent,
 *                                    boolean parentsFirst,
 *                                    boolean parentsRestrictedToBase)
 * </UL>  
 * 
 * <P>
 * Examples 1, 2, and 4 produce the same results: the same parent value and its 
 * children. 
 * Example 3 has two parents values and their children. 
 * Example 5 uses a parent value from a higher level ands sorts the parent and
 * child values hierarchically in descending order, with the parent values 
 * appearing after their children.
 * 
 * <P>
 * This program uses the Context10g class, which uses the
 * CursorPrintWriter class.
 **/
public class RecursiveJoinAndShortcutsExamples 
{
  public RecursiveJoinAndShortcutsExamples()
  {
  }
  
  public void run(String[] args) 
  {
    Context10g context = new Context10g(args, false);
    DataProvider dp = context.getDataProvider();
     
    // Get the MdmPrimaryDimension for the CUSTOMER dimension, 
    // the MdmLevelHierarchy for the SHIPMENTS_ROLLUP hierarchy of that
    // dimension, and the Source objects for them.
    MdmPrimaryDimension mdmCustDim =  
                        context.getMdmPrimaryDimensionByName("CUSTOMER");
    MdmLevelHierarchy mdmShipmentsRollupHier = (MdmLevelHierarchy) 
                    context.getHierarchyByName(mdmCustDim, "SHIPMENTS_ROLLUP");
    Source custDim = mdmCustDim.getSource();    
    StringSource shipmentsRollup = (StringSource) 
                                    mdmShipmentsRollupHier.getSource();
    
    // Get the parent attribute for the hierarchy and the value description
    // attribute for the dimension, and the Source objects for them.
    MdmAttribute mdmParentAttr = mdmShipmentsRollupHier.getParentAttribute();
    MdmAttribute mdmCustValDescAttr = mdmCustDim.getValueDescriptionAttribute();
    Source shipRollupParentAttr = mdmParentAttr.getSource();
    Source custValDescAttr = mdmCustValDescAttr.getSource();   
    
    // Specify a parent value from the hierarchy.
    Source parentValue = 
                shipmentsRollup.selectValue("SHIPMENTS_ROLLUP::WAREHOUSE::17");
    
    // Example 1: Using the full recursiveJoin method signature.
    // Create a Source that specifies the parent value and its children.
    Source parentAndChildren = 
                  shipmentsRollup.recursiveJoin(custDim.value(),
                                                parentValue,
                                                shipRollupParentAttr,
                                                Source.COMPARISON_RULE_SELECT,
                                                true,
                                                true,
                                                5,
                                                false);   
    // Create a Source that has the value descriptions for the parent and 
    // child values.
    Source parentAndChildrenWithDescr = custValDescAttr.join(parentAndChildren); 
    // Prepare and commit the Transaction.
    context.commit();  
    // Create a Cursor for parentAndChildrenWithDescr and display its values.
    context.println("\nUsing the full recursiveJoin method signature.");
    context.displayResult(parentAndChildrenWithDescr);
  
    
    // Example 2: recursiveJoin(Source joined, String comparison,  
    //                          Source parent, int comparisonRule)
    // Using a single String as the comparison, select the parent values and
    // their children.
    Source parentAndChildrenShortcut = 
                shipmentsRollup.recursiveJoin(custDim.value(),
                                              "SHIPMENTS_ROLLUP::WAREHOUSE::17",
                                               shipRollupParentAttr,
                                               Source.COMPARISON_RULE_SELECT);
    // Add the value descriptions to the selections.
    Source parentAndChildrenShortcutWithDescr = 
                               custValDescAttr.join(parentAndChildrenShortcut);
    context.commit();
    context.println("\nUsing the recursiveJoin(Source joined, " +
                    "String comparison,\n" + 
                    "Source parent, int comparisonRule) method.");
    context.displayResult(parentAndChildrenShortcutWithDescr);

    
    // Example 3: recursiveJoin(Source joined, String[] comparison, 
    //                          Source parent, int comparisonRule)
    // Using a String array as the comparison, select the parent values and
    // their children.   
    Source parentsAndChildrenShortcut = 
              shipmentsRollup.recursiveJoin(custDim.value(),
                                            new String[] 
                                            {"SHIPMENTS_ROLLUP::WAREHOUSE::17",
                                             "SHIPMENTS_ROLLUP::WAREHOUSE::18"},
                                             shipRollupParentAttr,
                                             Source.COMPARISON_RULE_SELECT);
    // Add the value descriptions to the selections.
    Source parentsAndChildrenShortcutWithDescr = 
                               custValDescAttr.join(parentsAndChildrenShortcut);    
    context.commit();
    context.println("\nUsing the recursiveJoin(Source joined, " +
                    "String[] comparison,\n" + 
                    "Source parent, int comparisonRule) method.");
    context.displayResult(parentsAndChildrenShortcutWithDescr);
    
    
    // Example 4: selectDescendants(Source comparison, Source parent)
    // Select the parent value and its descendants.
    Source parentAndChildrenShortcut2 = 
          shipmentsRollup.selectDescendants(
                    dp.createConstantSource("SHIPMENTS_ROLLUP::WAREHOUSE::17"),
                    shipRollupParentAttr);
    // Add the value descriptions to the selections.
    Source parentAndChildrenShortcut2WithDescr = 
                               custValDescAttr.join(parentAndChildrenShortcut2);    
    context.commit();
    context.println("\nUsing the selectDescendants(Source comparison, " +
                    "Source parent) method.");
    context.displayResult(parentAndChildrenShortcut2WithDescr);
     
     
    // Example 5: sortDescendingHierarchically(Source joined, Source parent,
    //                                         boolean parentsFirst,
    //                                         boolean parentsRestrictedToBase)
    // Select the descendants of a REGION value, which is the level above the
    // WAREHOUSE level.
    Source custSel = shipmentsRollup.selectDescendants(
                    dp.createConstantSource("SHIPMENTS_ROLLUP::REGION::8"),
                    shipRollupParentAttr);
    // Sort the selection hierarchically in descending order, with the parent
    // values appearing after their children.
    Source custSelSortedDescending = custSel.sortDescendingHierarchically(
                                              custDim.value(),
                                              shipRollupParentAttr,
                                              false, // parents after children
                                              true);
    // Add the value descriptions to the sorted selections.
    Source custSelSortedDescendingWithDescr = 
                               custValDescAttr.join(custSelSortedDescending);
    context.commit();
    context.println("\nUsing the sortDescendingHierarchically(Source joined, " +
                    "Source parent,\nboolean parentsFirst, " + 
                    "boolean parentsRestrictedToBase) method.");
    context.displayResult(custSelSortedDescendingWithDescr);
    
  }
  
  public static void main(String[] args)
  {
    new RecursiveJoinAndShortcutsExamples().run(args);
  }
}


